/*******************(C) COPYRIGHT 2025 Masses-Chip ******************************
* Project Name        : M9F6601_LCD.Prj
* File Name          : MAIN.C
* Author             : MASSES CHIP
* Version            : V1.01
* Date               : 2025/08/15
* Web    			 : www.masses-chip.com
* FAE                : Luo
*  				 	 : QQ：411680975
********************************************************************************
**注意：本范例仅供参考，实际应用需评估硬件兼容性
*Option:32M/4T,WDT-Enable-4s,RST-IO,BOR 2.5V 
*说明：本范例实现分钟/秒计时并通过LCD显示
*   - SEG端口：IOB口（通过SEGIOSB配置为SEG输出）
*   - COM端口：COM0-3（通过LCDCR0配置）
*   - 定时器0：用于产生1ms计时基准
******************************************************************************/
#include "zc.h"
#include "MASSESCHIP_DEFINE.H"
/**************************************************************************
* 段码表：共阴极LCD数码管编码
* 格式：0x7D -> "0"，0x05 -> "1"，以此类推
**************************************************************************/
u8 Coding_Tab[] = {
    0x7D,  // 0: 段a+b+c+d+f+g亮
    0x05,  // 1: 段b+c亮
    0x5B,  // 2: 段a+b+d+e+g亮
    0x1F,  // 3: 段a+b+c+d+g亮
    0x27,  // 4: 段b+c+f+g亮
    0x3E,  // 5: 段a+c+d+f+g亮
    0x7E,  // 6: 段a+c+d+e+f+g亮
    0x15,  // 7: 段a+b+c亮
    0x7F,  // 8: 全亮
    0x3F   // 9: a+b+c+d+f+g亮
};	
/**************************************************************************
* 全局变量定义
**************************************************************************/
u16 R_1ms_Count;       // 1ms计数器
u8 	Sec;               // 秒（0-59）
u8 	Min;               // 分（0-59）
u8 	LCD_Buf;           // LCD数据缓冲
u8 	LCD_Data1;         // 分钟十位
u8 	LCD_Data2;         // 分钟个位
u8 	LCD_Data3;         // 秒十位
u8 	LCD_Data4;         // 秒个位

ab8 R_Work_Flag;                 // 工作标志位集合
// 标志位定义
#define b_1S_F          R_Work_Flag.oneBit.b0  // 1秒触发标志

/**************************************************************************
* 函数声明
**************************************************************************/
void SYSTEM_INIT(void);      // 系统初始化
void LED_INIT(void);         // LCD模块初始化
void TC0_INIT(void);         // 定时器0初始化（1ms基准）
void LCD_Dis(void);          // LCD显示函数
//===============================================================================
//**********************************程序主函数***********************************
//===============================================================================
void main(void)						
{
	SYSTEM_INIT();   			// 系统初始化           	
	LED_INIT();					// LCD初始化				
	TC0_INIT();					// 定时器初始化
	while(1)
	{
		CLRWDT();				//清除看门狗
		if(b_1S_F == 1)			// 检测1秒标志
		{
			b_1S_F = 0;
			Sec++;
			if(Sec > 59)
			{
				Sec = 0;
				Min++;
				if(Min > 59)
				{
					Min = 0;				
				}
			}
		}

		// 更新显示数据
        LCD_Data1 = (u8)(Min / 10);   // 分钟十位
        LCD_Data2 = (u8)(Min % 10);   // 分钟个位
        LCD_Data3 = (u8)(Sec / 10);   // 秒十位
        LCD_Data4 = (u8)(Sec % 10);   // 秒个位
        
        LCD_Dis();              	  // 刷新LCD显示
	}
}

void LED_INIT(void)
{
//1、LCD端口设置
	OEB = 0x00;								//LCD模式下,用作LCD的端口需要配置为输入
	SEGIOSB = 0xFF;							//用作LED SEG输出
//2、LCD设置
	LCDCR0 = Bin(10001011);					//开LCD、B类模式、1/3BIAS、COM0-3作为COM口
	LCDCR1 = 0X2A;							//帧频
	LCDCR2 = Bin(00001110);					//小电阻开启时间
	LCDCR3 = Bin(00110000);					//VLCD电压
	LCDCR4 = Bin(00000011);					//OP
}

/**************************************************************************
* LCD显示函数
* 功能：将时间数据转换为段码并写入LCD数据寄存器
**************************************************************************/
void LCD_Dis()
{
// 1. 显示分钟十位（LCD_Data1）
	LCD_Buf = Coding_Tab[LCD_Data1];		
	LCDDS16 = (u8)((LCD_Buf>>4)&0x0F);
	LCDDS17 = (u8)(LCD_Buf&0x0F);
// 2. 显示分钟个位（LCD_Data2）
	LCD_Buf = Coding_Tab[LCD_Data2];		
	LCDDS18 = (u8)((LCD_Buf>>4)&0x0F);
	LCDDS19 = (u8)(LCD_Buf&0x0F);
// 3. 显示秒十位（LCD_Data3）
	LCD_Buf = Coding_Tab[LCD_Data3];		
	LCDDS20 = (u8)((LCD_Buf>>4)&0x0F);
	LCDDS21 = (u8)(LCD_Buf&0x0F);
// 4. 显示秒个位（LCD_Data4）
	LCD_Buf = Coding_Tab[LCD_Data4];		
	LCDDS22 = (u8)((LCD_Buf>>4)&0x0F);
	LCDDS23 = (u8)(LCD_Buf&0x0F);

}
/**************************************************************************
* 定时器TC0初始化（1ms中断）
* 配置说明：
  - 时钟源：FOSCH（32MHz）
  - 预分频：1:128（TC0CKPS=111）
  - 8位模式：TC0CH为计数器，TC0CL为周期值（计数到TC0CL时溢出）
  - 溢出时间：(TC0CL+1) * 预分频 / FOSCH = (249+1)*128/32MHz = 1ms
  - 中断：使能TC0溢出中断，高优先级
**************************************************************************/
void TC0_INIT()
{
	T0CR = Bin(10101111);		//8位模式，高频32M
	TC0CL = 249;				//(249+1)/(0.03125*128)=1MS
	TC0CH = 0;
	TC0IE = 1;
	TC0IF = 0;
	GIEH = 1;
}

//===============================================================================
//**************************高优先级中断服务子程序*******************************
//===============================================================================
void interrupt high_priority int_h(void)
{						
	if(TC0IF)
	{
		TC0IF = 0;
		R_1ms_Count++;
		if(R_1ms_Count == 1000)
		{
			R_1ms_Count=0;
			b_1S_F = 1;
		}
	}
}

//===============================================================================
//**************************低优先级中断服务子程序*******************************
//===============================================================================
void interrupt low_priority int_l(void)
{
 // 预留低优先级中断处理
}

//===============================================================================
//**********************************系统初始化***********************************
//系统时钟初始化：快钟、慢钟都正常工作，不进休眠
//IO初始化:设置IO状态为输出低，且关闭上下拉
//中断总开关：关闭总开关
//===============================================================================	
void SYSTEM_INIT(void)
{ 
//----------------------------------------------------
//Init 系统时钟
	OSCM = 0x00;
//----------------------------------------------------	            
//Init IO
	//IOA
	ANSA = 0x00;               //0:数字口	1:模拟口 
	PUA = 0x00;                //0:上拉关闭 1:上拉使能
	PDA = 0x00;				   //0:下拉关闭 1:下拉使能
	IOA = 0x00;                //IO口数据
	IOAOR = 0x00;			   //IO输出锁存
	OEA = 0xFF;                //0:输入 1:输出
	//IOB
	ANSB = 0x00;  
	PUB = 0x00;                
	PDB = 0x00;				   
	IOB = 0x00; 
	IOBOR = 0x00;
	OEB = 0xFF;                
	//IOC
	ANSC = 0x00;  
	PUC = 0x00;               
	PDC = 0x00;				   
	IOC = 0x00;
	IOCOR = 0x00;
	OEC = 0xFF;              
	//IOD
	ANSD = 0x00;  
	PUD = 0x00;               
	PDD = 0x00;				   
	IOD = 0x00; 
	IODOR = 0x00;
	OED = 0xFF;
	//IOE
	ANSE = 0x00;  
	PUE = 0x00;               
	PDE = 0x00;				   
	IOE = 0x00; 
	IOEOR = 0x00;
	OEE = 0xFF;
	//IOF
	ANSF = 0x00;  
	PUF = 0x00;               
	PDF = 0x00;				   
	IOF = 0x00;  
	IOFOR = 0x00;
	OEF = 0xFF;
//----------------------------------------------------     
//Init Interrupt
	GIEH = 0;                  //1:全局中断高优先级中断使能,0:屏蔽全局中断高优先级中断
	GIEL = 0;                  //1:全局中断低优先级中断使能,0:屏蔽全局中断低优先级中断
}

/*********************************END OF FILE************************************/